package android.dev.base.storage;

import android.content.Context;
import android.content.SharedPreferences;
import android.dev.base.components.AppContext;

import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * Created by oeager on 2016/1/20.
 * email:oeager@foxmail.com
 */
public final class PreferenceData {

    private final String fileName;

    private static volatile PreferenceData mInstance = null;

    private final SharedPreferences preferences;

    private PreferenceData() {
        //no instance
        AppContext app = AppContext.getApp();
        String packageName = app.getPackageName();
        fileName = Pattern.compile(".").matcher(packageName).replaceAll("_") + "-sp";
        preferences = app.getSharedPreferences(fileName, Context.MODE_PRIVATE);
    }

    public static PreferenceData getInstance() {
        if (mInstance == null) {
            synchronized (PreferenceData.class) {
                if (mInstance == null) {
                    mInstance = new PreferenceData();
                }
            }
        }
        return mInstance;
    }

    public <T> void save(String key, T value) {
        SharedPreferences.Editor editor = preferences.edit();
        editor.putString(key, String.valueOf(value));
        editor.apply();
    }

    @SuppressWarnings("unchecked")
    public <T> T take(String key, T defaultValue) {
        String s = preferences.getString(key, String.valueOf(defaultValue));
        if (defaultValue instanceof String) {
            return (T) s;
        } else if (defaultValue instanceof Integer) {
            return (T) Integer.valueOf(s);
        } else if (defaultValue instanceof Boolean) {
            return (T) Boolean.valueOf(s);
        } else if (defaultValue instanceof Float) {
            return (T) Float.valueOf(s);
        } else if (defaultValue instanceof Long) {
            return (T) Long.valueOf(s);
        }
        return null;
    }

    public void save(String key, Set<String> set) {
        SharedPreferences.Editor editor = preferences.edit();
        editor.putStringSet(key, set);
        editor.apply();
    }

    public Set<String> take(String key, Set<String> defaultValue) {
        return preferences.getStringSet(key, defaultValue);
    }

    public void remove(String key) {
        SharedPreferences.Editor editor = preferences.edit();
        editor.remove(key);
        editor.apply();
    }

    public void clear() {
        SharedPreferences.Editor editor = preferences.edit();
        editor.clear();
        editor.apply();
    }

    public boolean contains(String key) {
        return preferences.contains(key);
    }

    public Map<String, ?> getAll() {
        return preferences.getAll();
    }


}
    