package android.dev.base.arch;

import android.app.Application;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by wbs on 2018/3/27 0027.
 */

public final class ModuleLoader implements Pool<Module> {

    private ModuleLoader() {
    }

    private final static ModuleLoader _IMPL = new ModuleLoader();

    public static ModuleLoader getDefault() {
        return _IMPL;
    }

    private final Map<String, Module> mModules = new HashMap<>();

    public void register(Module module) {
        if (module == null) {
            return;
        }
        synchronized (mModules) {
            mModules.put(module.getName(), module);
        }
    }

    @Override
    public void unregister(Module module) {
        if (module == null) {
            return;
        }
        synchronized (mModules) {
            mModules.remove(module.getName());
        }
    }

    @Override
    public void clear() {
        synchronized (mModules) {
            mModules.clear();
        }
    }


    final void performInstalled(Application app) {
        synchronized (mModules) {
            for (Module module : mModules.values()) {
                module.install(app);
            }
        }
    }

    final void performUninstalled(Application app) {
        synchronized (mModules) {
            for (Module module : mModules.values()) {
                module.uninstall(app);
            }
        }
    }

    public Module getModule(String name) {
        synchronized (mModules) {
            return mModules.get(name);
        }
    }

    @SuppressWarnings("unchecked")
    public <M extends Module> M getModule(String name, Class<M> cls) {
        Module m = getModule(name);
        if (m == null) {
            return null;
        }
        if (m.getClass() == cls) {
            return (M) m;
        }
        return null;
    }

    public int getModuleCount(){
        return mModules.size();
    }

}
