package android.dev.base.components;

import android.app.Activity;
import android.app.Application;
import android.app.Dialog;
import android.content.res.Configuration;
import android.dev.base.api.container.ActionBarContainer;
import android.dev.base.api.injector.ActionBarInjector;
import android.dev.base.api.injector.ActivityLayoutLayerInjector;
import android.dev.base.api.injector.FragmentLayoutLayerInjector;
import android.dev.base.api.injector.LayoutLayerInjector;
import android.dev.base.api.option.MultiStateConfigurator;
import android.dev.base.api.ProgressDialogConstructor;
import android.dev.base.api.option.SwipeBackConfigurator;
import android.dev.base.apiImpl.MultiStateLayerInjectorActivityImpl;
import android.dev.base.apiImpl.MultiStateLayerInjectorFragmentImpl;
import android.dev.base.apiImpl.SwipeBackLayerInjector;
import android.dev.base.view.DefaultProgressDialog;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Created by wbs on 2017/11/17 0017.
 */

public final class ComponentsBox {

    private ComponentsBox() {
        registerLayoutLayerInjector(new MultiStateLayerInjectorActivityImpl());
        registerLayoutLayerInjector(new SwipeBackLayerInjector());
        registerLayoutLayerInjector(new MultiStateLayerInjectorFragmentImpl());
    }

    private static class Holder {
        private final static ComponentsBox IMPL = new ComponentsBox();
    }

    public static ComponentsBox getDefault() {
        return Holder.IMPL;
    }

    //--------------------------AppDeck Inject----------------------------
    private final List<ApplicationDeck> mAppDecks = new ArrayList<>();

    public int getAppDeckSize() {
        return mAppDecks.size();
    }

    public void injectAppDeck(AppDeck deck) {
        synchronized (mAppDecks) {
            mAppDecks.add(deck);
        }
    }

    public void performAppOnCreated(Application app) {
        for (ApplicationDeck deck : mAppDecks) {
            deck.onCreate(app);
        }
    }

    public void performAppTerminate(Application app) {
        for (ApplicationDeck deck : mAppDecks) {
            deck.onTerminate(app);
        }
    }

    public void performAppOnConfigurationChanged(Application app, Configuration newConfig) {
        for (ApplicationDeck deck : mAppDecks) {
            deck.onConfigurationChanged(app, newConfig);
        }
    }

    public void performOnLowMemory(Application application) {
        for (ApplicationDeck deck : mAppDecks) {
            deck.onLowMemory(application);
        }
    }

    public void performOnTrimMemory(Application application, int level) {
        for (ApplicationDeck deck : mAppDecks) {
            deck.onTrimMemory(application, level);
        }
    }


    @SuppressWarnings("unchecked")
    @Nullable
    public <A extends ApplicationDeck> A getAppDeck(Class<? extends ApplicationDeck> cls) {
        synchronized (mAppDecks) {
            if (mAppDecks.isEmpty()) {
                return null;
            }
            for (ApplicationDeck deck : mAppDecks) {
                if (deck.getClass() == cls) {
                    return (A) deck;
                }
            }
        }
        return null;
    }

    //-----------------------App ProgressDialog Inject------------------------

    private ProgressDialogConstructor mDialogConstructor;

    public Dialog createProgressDialog(Activity activity) {
        if (mDialogConstructor == null) {
            mDialogConstructor = new ProgressDialogConstructor() {
                @Override
                public Dialog construct(Activity activity) {
                    return new DefaultProgressDialog(activity);
                }
            };
        }
        return mDialogConstructor.construct(activity);
    }

    public void registProgressDialogConstructor(ProgressDialogConstructor constructor) {
        this.mDialogConstructor = constructor;
    }
    //---------------------App call to Inject---------------------------------

    private MultiStateConfigurator mMultiStateConfigurator;

    private SwipeBackConfigurator mSwipeBackConfigurator;


    public void registMultiStateInjector(MultiStateConfigurator configurator) {
        mMultiStateConfigurator = configurator;
    }

    public void registSwipeBackConfigurator(SwipeBackConfigurator configurator) {
        mSwipeBackConfigurator = configurator;
    }

    public MultiStateConfigurator getMultiStateConfigurator() {
        return mMultiStateConfigurator;
    }

    public SwipeBackConfigurator getSwipeBackConfigurator() {
        return mSwipeBackConfigurator;
    }

    //----------------------App Layout inject-----------

    private final List<ActivityLayoutLayerInjector> mLayoutInjectors = new ArrayList<>();

    private final List<FragmentLayoutLayerInjector> mLayoutInjectors2 = new ArrayList<>();

    public void registerLayoutLayerInjector(LayoutLayerInjector injector) {
        if (injector instanceof ActivityLayoutLayerInjector) {
            synchronized (mLayoutInjectors) {
                mLayoutInjectors.add((ActivityLayoutLayerInjector) injector);
            }
        } else if (injector instanceof FragmentLayoutLayerInjector) {
            synchronized (mLayoutInjectors2) {
                mLayoutInjectors2.add((FragmentLayoutLayerInjector) injector);
            }
        }

    }

    public void unregisterLayoutLayersInjector(Class<? extends LayoutLayerInjector> cls) {

        synchronized (mLayoutInjectors) {
            Iterator<ActivityLayoutLayerInjector> iterator = mLayoutInjectors.iterator();
            while (iterator.hasNext()) {
                ActivityLayoutLayerInjector injector = iterator.next();
                if (injector.getClass().equals(cls)) {
                    iterator.remove();
                    return;
                }
            }
        }
        synchronized (mLayoutInjectors2) {
            Iterator<FragmentLayoutLayerInjector> iterator = mLayoutInjectors2.iterator();
            while (iterator.hasNext()) {
                FragmentLayoutLayerInjector injector = iterator.next();
                if (injector.getClass().equals(cls)) {
                    iterator.remove();
                    return;
                }
            }
        }


    }

    public void performInjectLayoutLayers(Activity activity) {
        synchronized (mLayoutInjectors) {
            for (ActivityLayoutLayerInjector injector : mLayoutInjectors) {
                injector.injectLayout(this, activity);
            }
        }
    }

    public void performInjectLayoutLayers(Fragment fragment) {
        synchronized (mLayoutInjectors2) {
            for (FragmentLayoutLayerInjector injector : mLayoutInjectors2) {
                injector.injectLayout(this, fragment);
            }
        }
    }

    //------------------------inject ActionBar---------------------

    private ActionBarInjector mActionBarInjector;

    public void registerActionBarInjector(ActionBarInjector injector) {
        this.mActionBarInjector = injector;
    }

    public void performInjectActionBarLayout(Activity activity) {
        if (mActionBarInjector == null) {
            return;
        }
        if (activity instanceof ActionBarContainer) {
            mActionBarInjector.injectActionBarLayout(activity);
        }
    }
}
