package android.dev.base.components;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.res.Configuration;
import android.os.Build;
import android.os.Bundle;
import android.os.StrictMode;

import java.util.Iterator;
import java.util.Stack;

/**
 * Created by wbs on 2017/12/7 0007.
 */

public final class AppContextLike implements ApplicationDeck,Application.ActivityLifecycleCallbacks {

    private final static AppContextLike IMPL = new AppContextLike();

    public static AppContextLike getDefault(){
        return IMPL;
    }

    //Don't care
    @SuppressLint("StaticFieldLeak")
    private volatile static Application mDefault;

    private final Stack<Activity> activityStack = new Stack<>();

    private AppContextLike(){

    }

    public void setApplication(Application application){
        mDefault = application;
    }

    @Override
    public void onCreate(Application application) {
        if(mDefault==null){
            mDefault = application;
        }
        if (AppEnvironment.DEBUG) {
            StrictMode.VmPolicy.Builder builder = new StrictMode.VmPolicy.Builder()
                    .detectActivityLeaks()   //检测Activity泄漏
                    .detectLeakedSqlLiteObjects()//检测数据库泄漏
                    .detectLeakedClosableObjects();//检测Closeable对象泄漏
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                builder.detectLeakedRegistrationObjects();//检测注册对象的泄漏
            }

            StrictMode.setVmPolicy(builder.penaltyLog().build());
        }
        application.registerActivityLifecycleCallbacks(this);
        ComponentsBox.getDefault().performAppOnCreated(application);
    }

    @Override
    public void onTerminate(Application application) {
        try {
            finishAllActivity();
            application.unregisterActivityLifecycleCallbacks(this);
            android.os.Process.killProcess(android.os.Process.myPid());
            System.exit(0);

        } catch (Exception ignored) {
        }
        ComponentsBox.getDefault().performAppTerminate(application);

    }

    @Override
    public void onConfigurationChanged(Application application, Configuration newConfig) {
        ComponentsBox.getDefault().performAppOnConfigurationChanged(application, newConfig);
    }

    @Override
    public void onLowMemory(Application application) {
        ComponentsBox.getDefault().performOnLowMemory(application);
    }

    @Override
    public void onTrimMemory(Application application, int level) {
        ComponentsBox.getDefault().performOnTrimMemory(application, level);
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        joinActivity(activity);
    }

    @Override
    public void onActivityStarted(Activity activity) {

    }

    @Override
    public void onActivityResumed(Activity activity) {

    }

    @Override
    public void onActivityPaused(Activity activity) {

    }

    @Override
    public void onActivityStopped(Activity activity) {

    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        popActivity(activity);
    }
    //------------------------------------------private method-----------------------
    private void joinActivity(Activity activity) {
        activityStack.add(activity);
    }

    private void popActivity(Activity activity) {
        if (activity != null) {
            this.activityStack.remove(activity);
        }
    }
    //---------------------------------------package method---------------------
    final void finishActivity() {
        if (activityStack.isEmpty()) {
            return;
        }
        Activity activity = activityStack.pop();
        if (activity != null && !activity.isFinishing()) {
            activity.finish();
        }
    }

    final void finishActivity(Activity activity) {
        if (activity != null) {
            this.activityStack.remove(activity);
            if (!activity.isFinishing()) {
                activity.finish();
            }
        }
    }

    final void finishActivity(Class<? extends Activity> cls) {
        Iterator<Activity> iterator = activityStack.iterator();
        while (iterator.hasNext()) {
            Activity activity = iterator.next();
            if (activity.getClass().equals(cls)) {
                iterator.remove();
                if (!activity.isFinishing()) {
                    activity.finish();
                }
            }
        }
    }

    final void finishAllActivity() {
        while (!activityStack.isEmpty()) {
            Activity activity = activityStack.pop();
            if (activity != null && !activity.isFinishing()) {
                activity.finish();
            }
        }
    }

    final Activity getActivity(Class<? extends Activity> cls) {
        for (Activity activity : activityStack) {
            if (activity.getClass().equals(cls)) {
                return activity;
            }
        }
        return null;
    }

    final Activity currentActivity() {
        if (activityStack.isEmpty()) {
            return null;
        }
        return activityStack.peek();
    }

    final void appExit(boolean killProcess) {
        try {
            finishAllActivity();
            if (killProcess) {
                if(mDefault!=null) {
                    mDefault.unregisterActivityLifecycleCallbacks(this);
                }
                android.os.Process.killProcess(android.os.Process.myPid());
                System.exit(0);
            }
        } catch (Exception ignored) {
        }
    }

    final String dumpStackInfo() {
        StringBuilder stringBuilder = new StringBuilder();
        for (Activity activity : activityStack) {
            stringBuilder.append(activity.getClass().getName()).append("\n");
        }
        return stringBuilder.toString();
    }

    final int getStackCount() {
        return activityStack.size();
    }

    static  <A extends Application> A getAppContext(){
        if(mDefault==null){
            return null;
        }
        return (A) mDefault;
    }
}
