package android.dev.compiler;

import android.dev.annotation.Forward;
import android.dev.annotation.Link;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedOptions;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.MirroredTypeException;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.ElementFilter;
import javax.tools.Diagnostic;

/**
 * Created by wbs on 2017/11/29 0029.
 */

@SupportedOptions("host")
public class RouterProcessor extends BaseProcessor {

    private final static String APT_SOURCE_PKG = "android.dev.apt.router";

    private static final String ACTIVITY = "android.app.Activity";

    private static final String FRAGMENT = "android.app.Fragment";

    private static final String FRAGMENT_V4 = "android.support.v4.app.Fragment";

    private final static String JAVA_DOC = "@ automatically generated by Apt\n";

    private final static String CLASS_BRIDGE_PATH = "android.dev.router.Bridge";

    private String mHost;

    @Override
    public synchronized void init(ProcessingEnvironment processingEnvironment) {
        super.init(processingEnvironment);
        Map<String, String> options = processingEnv.getOptions();
        if (options != null && !options.isEmpty()) {
            mHost = options.get("host");
        }
    }

    @Override
    public boolean processing(Set<? extends TypeElement> set, RoundEnvironment roundEnvironment) {
        if (mHost != null && mHost.length() > 0) {
            String[] pots = mHost.split("\\.");
            StringBuilder builder = new StringBuilder();
            for (String pot : pots) {
                builder.append(pot).append("_");
            }
            builder.append("Bridge$Impl");
            mHost = builder.toString().replaceAll("[^0-9a-zA-Z_]+", "");
            log("The user has configuration the mHost name, it was [" + mHost + "]");
        } else {
            error("These no mHost name, at 'build.gradle', like :\n" +
                    "javaCompileOptions {\n" +
                    "            annotationProcessorOptions {\n" +
                    "                arguments = [ host : 'your mHost name,may be use[project.getName()]' ]\n" +
                    "            }\n" +
                    "        }");
        }
        log("start generate Bridge Impl class:[" + mHost + "]");
        TypeMirror typeActivity = mElements.getTypeElement(ACTIVITY).asType();
        TypeMirror typeFragment = mElements.getTypeElement(FRAGMENT).asType();
        TypeMirror typeFragmentV4 = mElements.getTypeElement(FRAGMENT_V4).asType();
        TypeMirror typeBridge = mElements.getTypeElement(CLASS_BRIDGE_PATH).asType();

        TypeSpec.Builder classBuilder = TypeSpec.classBuilder(mHost)
                .addJavadoc(JAVA_DOC)
                .addSuperinterface(TypeName.get(typeBridge))
                .addModifiers(Modifier.PUBLIC, Modifier.FINAL);

        ClassName clsAddress = ClassName.get("android.dev.router", "Address");

        TypeName mapType = ParameterizedTypeName.get(
                ClassName.get(Map.class),
                TypeName.get(String.class),
                clsAddress);

        classBuilder.addField(mapType, "mTable", Modifier.PRIVATE, Modifier.FINAL);

        classBuilder.addInitializerBlock(CodeBlock.builder()
                .addStatement("mTable = new $T<>()", ClassName.get(HashMap.class))
                .build());

        MethodSpec.Builder methodGetHost = MethodSpec.methodBuilder("getHost")
                .addAnnotation(Override.class)
                .addJavadoc(JAVA_DOC)
                .addModifiers(Modifier.PUBLIC)
                .returns(String.class);

        methodGetHost.addStatement(" return $S", mHost);

        classBuilder.addMethod(methodGetHost.build());

        MethodSpec.Builder methodSearchAddress = MethodSpec.methodBuilder("searchAddress")
                .addJavadoc(JAVA_DOC)
                .addAnnotation(Override.class)
                .addParameter(String.class, "path")
                .addModifiers(Modifier.PUBLIC)
                .returns(clsAddress);
        methodSearchAddress.addStatement("return mTable.get(path)");
        classBuilder.addMethod(methodSearchAddress.build());

        MethodSpec.Builder methodGenerateSpaceAddress = MethodSpec.methodBuilder("generateSpaceAddress")
                .addJavadoc(JAVA_DOC)
                .addModifiers(Modifier.PRIVATE)
                .returns(clsAddress);
        methodGenerateSpaceAddress.addStatement("$T address = new $T()", clsAddress, clsAddress);
        methodGenerateSpaceAddress.addStatement("address.mHost = getHost()");
        methodGenerateSpaceAddress.addStatement("return address");

        classBuilder.addMethod(methodGenerateSpaceAddress.build());
        List<RouterData> mData = new ArrayList<>();
        for (TypeElement typeElement : ElementFilter.typesIn(roundEnvironment.getElementsAnnotatedWith(Link.class))) {
            TypeMirror tm = typeElement.asType();
            Link link = typeElement.getAnnotation(Link.class);
            RouterData router = new RouterData();
            String value = link.value();
            if (!value.startsWith("/")) {
                error("link means the path of the uri,must start with '/' ->[" + value + "]");
            }
            router.value = link.value();
            router.flag = link.flag();
            Forward forward = typeElement.getAnnotation(Forward.class);
            if (mType.isSubtype(tm, typeActivity)) {
                router.mType = RouterData.TYPE_ACTIVITY;
                router.mActivityElement = typeElement.getQualifiedName().toString();
                router.mFragmentElement = null;
            } else if (mType.isSubtype(tm, typeFragment)) {
                router.mType = RouterData.TYPE_FRAGMENT;
                router.mFragmentElement = typeElement;
                if (forward == null) {
                    error("Fragment annotated @Link,must annotated @Forward to set the Activity");
                    return false;
                }
                String name = forward.name();
                if(name.length()>0){
                    router.mActivityElement = name;
                }else {
                    TypeElement activityElement = getAnnotationClassElement(forward);
                    if (!mType.isSubtype(activityElement.asType(), typeActivity)) {
                        error("Fragment annotated @Forward,must be a subclass of Activity");
                    }
                    router.mActivityElement = activityElement.getQualifiedName().toString();
                }
            } else if (mType.isSubtype(tm, typeFragmentV4)) {
                router.mType = RouterData.TYPE_FRAGMENT_V4;
                router.mFragmentElement = typeElement;
                if (forward == null) {
                    error("Fragment annotated @Link,must annotated @Forward to set the Activity");
                    return false;
                }
                String name = forward.name();
                if(name.length()>0){
                    router.mActivityElement = name;
                }else {
                    TypeElement activityElement = getAnnotationClassElement(forward);
                    if (!mType.isSubtype(activityElement.asType(), typeActivity)) {
                        error("Fragment annotated @Forward,must be a subclass of Activity");
                    }
                    router.mActivityElement = activityElement.getQualifiedName().toString();
                }

            } else {
                error("Annotation [Link] only used for class activity or fragment,but you used for:" + tm.toString());
            }
            mData.add(router);
        }

        MethodSpec.Builder constructorMethod = MethodSpec.constructorBuilder()
                .addModifiers(Modifier.PUBLIC);
        CodeBlock.Builder constructCode = CodeBlock.builder();
        if (mData.size() > 0) {
            RouterData data = mData.get(0);
            constructCode.addStatement("$T address = generateSpaceAddress()", clsAddress);
            addRouterStatement(constructCode, data);
        }

        if (mData.size() > 1) {
            for (int i = 1; i < mData.size(); i++) {
                RouterData data = mData.get(i);
                constructCode.addStatement("address = generateSpaceAddress()");
                addRouterStatement(constructCode, data);
            }
        }

        constructorMethod.addCode(constructCode.build());

        classBuilder.addMethod(constructorMethod.build());


        JavaFile javaFile = JavaFile.builder(APT_SOURCE_PKG, classBuilder.build()).build();
        try {
            javaFile.writeTo(mFiler);
        } catch (IOException e) {
            e.printStackTrace();
        }


        return true;
    }

    @Override
    protected void putSupportAnnotations(Set<String> types) {
        types.add(Link.class.getName());
        types.add(Forward.class.getName());
    }

    private void addRouterStatement(CodeBlock.Builder codeBuilder, RouterData routerData) {
        codeBuilder.addStatement("address.mPathName = $S",routerData.mActivityElement);
        String value = routerData.value;
        codeBuilder.addStatement("address.mLink = $S", value);
        codeBuilder.addStatement("address.mFlag = $L",routerData.flag);
        codeBuilder.addStatement("address.mType = $L",routerData.mType);
        if (routerData.mType == RouterData.TYPE_FRAGMENT_V4) {
            codeBuilder.addStatement("address.mFragmentName = $T.class.getName()",routerData.mFragmentElement);
        } else if (routerData.mType == RouterData.TYPE_FRAGMENT) {
            codeBuilder.addStatement("address.mFragmentName = $T.class.getName()",routerData.mFragmentElement);
        }
        if (value != null && value.length() > 0) {
            codeBuilder.addStatement("mTable.put($S,address)", value);
        } else {
            error("why path is null for:" + value);
        }
        codeBuilder.add("//---------------------------\n");

    }

    private TypeElement getAnnotationClassElement(Forward forward) {
        String name;
        try {
            Class<?> clazz = forward.value();
            name = clazz.getCanonicalName();
        } catch (MirroredTypeException e) {
            DeclaredType type = (DeclaredType) e.getTypeMirror();
            TypeElement element = (TypeElement) type.asElement();
            name = element.getQualifiedName().toString();
        }
        return mElements.getTypeElement(name);
    }

}
