package android.dev.compiler;

import android.dev.annotation.LinkQuery;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.element.Element;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;

/**
 * Created by wbs on 2017/12/5 0005.
 */

public class LinkQueryProcessor extends BaseProcessor {

    private static final String ACTIVITY = "android.app.Activity";

    private static final String FRAGMENT = "android.app.Fragment";

    private static final String FRAGMENT_V4 = "android.support.v4.app.Fragment";

    private static final String LINK_QUERY_INJECTOR = "android.dev.router.LinkQueryInjector";

    private final static String JAVA_DOC = "@ automatically generated by Apt\n";

    private final static String SUFFIX_INJECTOR = "$Injector";

    private final TypeName mExceptionType = TypeName.get(Exception.class);


    @Override
    protected boolean processing(Set<? extends TypeElement> set, RoundEnvironment roundEnvironment) {
        log("start process LinkQuery Annotations");
        Set<? extends Element> elementSet = roundEnvironment.getElementsAnnotatedWith(LinkQuery.class);
        final Map<TypeElement, List<Element>> parentAndChild = new HashMap<>();
        for (Element element : elementSet) {
            //获取所在类的TypeElement
            TypeElement enclosingElement = (TypeElement) element.getEnclosingElement();
            if (element.getModifiers().contains(Modifier.PRIVATE)) {
                error("The inject fields CAN NOT BE 'private'!!! please check field ["
                        + element.getSimpleName() + "] in class [" + enclosingElement.getQualifiedName() + "]");

            }
            List<Element> mElements = parentAndChild.computeIfAbsent(enclosingElement, k -> new ArrayList<>());
            mElements.add(element);
        }
        if (!parentAndChild.isEmpty()) {
            TypeMirror typeActivity = mElements.getTypeElement(ACTIVITY).asType();
            TypeMirror typeFragment = mElements.getTypeElement(FRAGMENT).asType();
            TypeMirror typeFragmentV4 = mElements.getTypeElement(FRAGMENT_V4).asType();
            TypeMirror typeInjector = mElements.getTypeElement(LINK_QUERY_INJECTOR).asType();
            ParameterSpec objectParamSpec = ParameterSpec.builder(TypeName.OBJECT, "target").build();
            for (Map.Entry<TypeElement, List<Element>> entry : parentAndChild.entrySet()) {
                TypeElement parent = entry.getKey();
                List<Element> children = entry.getValue();
                String qualifiedName = parent.getQualifiedName().toString();
                String packageName = qualifiedName.substring(0, qualifiedName.lastIndexOf("."));
                String fileName = parent.getSimpleName() + SUFFIX_INJECTOR;
                TypeSpec.Builder classBuilder = TypeSpec.classBuilder(fileName)
                        .addJavadoc(JAVA_DOC)
                        .addSuperinterface(ClassName.get(typeInjector))
                        .addModifiers(Modifier.PUBLIC);

                MethodSpec.Builder methodInject = MethodSpec.methodBuilder("inject")
                        .addAnnotation(Override.class)
                        .addJavadoc(JAVA_DOC)
                        .addParameter(objectParamSpec)
                        .addModifiers(Modifier.PUBLIC);

                methodInject.addStatement("$T _target = ($T)target", ClassName.get(parent), ClassName.get(parent));
                ClassName queryParserClazz = ClassName.get("android.dev.router", "QueryParser");
                TypeMirror parentType = parent.asType();
                boolean isActivity;
                if (mType.isSubtype(parentType, typeActivity)) {
                    methodInject.addStatement("$T data = _target.getIntent()", ClassName.get("android.content", "Intent"));
                    isActivity = true;
                } else if (mType.isSubtype(parentType, typeFragment) || mType.isSubtype(parentType, typeFragmentV4)) {
                    methodInject.addStatement("$T data = _target.getArguments()", ClassName.get("android.os", "Bundle"));
                    isActivity = false;
                } else {
                    return false;
                }
                JavaTypePool javaTypePool = new JavaTypePool(mType, mElements, getLogs());
                methodInject.addStatement(" String value = null");
                for (Element element : children) {
                    LinkQuery query = element.getAnnotation(LinkQuery.class);
                    String fieldName = element.getSimpleName().toString();
                    int type = javaTypePool.getTypeOrdinal(element);
                    String key = query.value();
                    if (type == TypeKind.BOOLEAN.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseBoolean","false"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.BYTE.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseByte","0"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.SHORT.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseShort","0"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.INT.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseInt","0"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.LONG.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseLong","0L"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.CHAR.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseChar","0"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.FLOAT.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseFloat","0f"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.DOUBLE.ordinal()) {
                        methodInject.addStatement(formatSupportNull(isActivity,"parseDouble","0.00"),fieldName,key,queryParserClazz);
                    } else if (type == TypeKind.STRING.ordinal()) {
                        methodInject.addStatement("_target.$L = " + getFormatExtraGetter(isActivity), fieldName, key);
                    } else if (type == TypeKind.BOOLEAN_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getBooleanArrayExtra($S)" : "_target.$L = data.getBooleanArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.INT_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getIntArrayExtra($S)" : "_target.$L = data.getIntArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.LONG_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getLongArrayExtra($S)" : "_target.$L = data.getLongArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.CHAR_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getCharArrayExtra($S)" : "_target.$L = data.getCharArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.FLOAT_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getFloatArrayExtra($S)" : "_target.$L = data.getFloatArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.DOUBLE_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getDoubleArrayExtra($S)" : "_target.$L = data.getDoubleArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.STRING_ARRAY.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getStringArrayExtra($S)" : "_target.$L = data.getStringArray($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.STRING_LIST.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getStringArrayListExtra($S)" : "_target.$L = data.getStringArrayList($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.PARCELABLE_LIST.ordinal()) {
                        String formatText = isActivity ? "_target.$L = data.getParcelableArrayListExtra($S)" : "_target.$L = data.getParcelableArrayList($S)";
                        addStatementSurroundException(methodInject,formatText, fieldName, key);
                    } else if (type == TypeKind.PARCELABLE.ordinal()) {
                        if (isActivity) {
                            addStatementSurroundException(methodInject,"_target.$L = data.getParcelableExtra($S)", fieldName, key);
                        } else {
                            addStatementSurroundException(methodInject,"_target.$L = data.getParcelable($S)", fieldName, key);
                        }
                    } else if (type == TypeKind.SERIALIZABLE.ordinal()) {
                        if (isActivity) {
                            addStatementSurroundException(methodInject,"_target.$L = ($T) data.getSerializableExtra($S)", fieldName, ClassName.get(element.asType()), key);
                        } else {
                            addStatementSurroundException(methodInject,"_target.$L = ($T) data.getSerializable($S)", fieldName, ClassName.get(element.asType()), key);
                        }
                    } else {
                        //todo nothing
                    }

                }
                classBuilder.addMethod(methodInject.build());

                try {
                    JavaFile.builder(packageName, classBuilder.build()).build().writeTo(mFiler);
                } catch (IOException e) {
                    e.printStackTrace();
                }

            }
        }

        return false;
    }

    private void addStatementSurroundException(MethodSpec.Builder specInjector, String format, Object... args) {

        specInjector.beginControlFlow("try")
                .addStatement(format, args)
                .endControlFlow()
                .beginControlFlow("catch($T ignore)",mExceptionType)
                .endControlFlow();
    }

    private String formatSupportNull(boolean isActivity, String parseChar,String def) {
        String formatGetter = getFormatExtraGetter(isActivity);
        return "_target.$L = (value=" + formatGetter + ")!=null?$T." + parseChar + "(value):"+def;
    }

    private String getFormatExtraGetter(boolean isActivity) {
        return isActivity ? "data.getStringExtra($S)" : "data.getString($S)";
    }


    @Override
    protected void putSupportAnnotations(Set<String> types) {
        types.add(LinkQuery.class.getName());
    }
}
