package com.tinchoapps.devutils.Camera;

import android.content.Context;
import android.hardware.Camera;
import android.support.annotation.NonNull;
import android.view.Display;
import android.view.Surface;
import android.view.WindowManager;

import java.util.List;

public class CameraUtils
{
    /**
     * Iterate over supported camera preview sizes to see which one best fits the
     * dimensions of the given view while maintaining the aspect ratio. If none can,
     * be lenient with the aspect ratio.
     *
     * @param camera Camera obj.
     * @param w      The width of the view.
     * @param h      The height of the view.
     * @return Best match camera preview size to fit in the view.
     */
    public static Camera.Size getOptimalPreviewSize(@NonNull final Camera camera, final int w, final int h)
    {
        List<Camera.Size> sizes = camera.getParameters().getSupportedPreviewSizes();

        // Use a very small tolerance because we want an exact match.
        final double ASPECT_TOLERANCE = 0.1;
        double targetRatio = (double) w / h;
        if (sizes == null)
            return null;

        Camera.Size optimalSize = null;

        // Start with max value and refine as we iterate over available preview sizes. This is the
        // minimum difference between view and camera height.
        double minDiff = Double.MAX_VALUE;

        // Try to find a preview size that matches aspect ratio and the target view size.
        // Iterate over all available sizes and pick the largest size that can fit in the view and
        // still maintain the aspect ratio.
        for (Camera.Size size : sizes)
        {
            double ratio = (double) size.width / size.height;
            if (Math.abs(ratio - targetRatio) > ASPECT_TOLERANCE)
                continue;
            if (Math.abs(size.height - h) < minDiff)
            {
                optimalSize = size;
                minDiff = Math.abs(size.height - h);
            }
        }

        // Cannot find preview size that matches the aspect ratio, ignore the requirement
        if (optimalSize == null)
        {
            minDiff = Double.MAX_VALUE;
            for (Camera.Size size : sizes)
            {
                if (Math.abs(size.height - h) < minDiff)
                {
                    optimalSize = size;
                    minDiff = Math.abs(size.height - h);
                }
            }
        }

        return optimalSize;
    }

    public static void setCameraPreviewOrientation(@NonNull Context context, @NonNull Camera camera)
    {
        Camera.Parameters parameters = camera.getParameters();
        Display display = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();

        if (display.getRotation() == Surface.ROTATION_0)
        {
            camera.setDisplayOrientation(90);
        } else if (display.getRotation() == Surface.ROTATION_270)
        {
            camera.setDisplayOrientation(180);
        }

        camera.setParameters(parameters);
    }

    public static void stopPreview(@NonNull Camera camera)
    {
        camera.stopPreview();
        camera.setPreviewCallback(null);
    }
}
