/*
 * Copyright 2015 Kim Seong-il
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kr.seongil.recyclerview.viewbinder;

import android.support.annotation.NonNull;
import android.support.v4.util.SparseArrayCompat;
import android.support.v7.widget.RecyclerView;
import android.util.Log;
import android.view.ViewGroup;

import java.util.List;

import kr.seongil.recyclerview.utils.RecyclerViewUtils;

/**
 * I refer to follow project and customized it.
 * https://github.com/sockeqwe/AdapterDelegates/
 * This class is the element that ties {@link RecyclerView} together with {@link ViewBinder}.
 * So you have to add / register your {@link ViewBinder}s to this manager by calling {@link #addViewBinder(ViewBinder)}
 *
 * @author Kim Seong-il
 * @see android.support.v7.widget.RecyclerView.Adapter
 * @since 1.0.0
 */
public class ViewBinderListManager<T extends List> {

    // ===========================================================
    // Constants
    // ===========================================================
    private static final String TAG = RecyclerViewUtils.TAG;
    private static final String LOG_PREFIX = RecyclerViewUtils.getPrefix("ViewBinderListManager");

    // ===========================================================
    // Fields
    // ===========================================================
    private SparseArrayCompat<ViewBinder<T>> mViewBinderList;

    // ===========================================================
    // Constructors
    // ===========================================================
    public ViewBinderListManager() {
        this(8);
    }

    public ViewBinderListManager(final int capacity) {
        if (capacity < 0) {
            throw new IllegalArgumentException("capacity < 0: " + capacity);
        }

        mViewBinderList = new SparseArrayCompat<>(capacity);
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    // ===========================================================
    // Methods
    // ===========================================================
    public boolean addViewBinder(@NonNull ViewBinder<T> viewBinder) {
        RecyclerViewUtils.checkNotNull(viewBinder, "ViewBinder is null. You must pass valid object.");

        final int viewType = viewBinder.getItemViewType();
        if (existViewBinder(viewBinder)) {
            StringBuilder sb = new StringBuilder(1024);
            sb.append(LOG_PREFIX + "ViewBinder is been registered in ViewBinderList already. type : " + viewType + '\n');
            if (viewType == AbstractViewBinder.RECYCLER_HEADER_VIEW_TYPE) {
                sb.append("Requested ViewType is HeaderView type. So, you have to use another type value.");
            } else if (viewType == AbstractViewBinder.RECYCLER_FOOTER_VIEW_TYPE) {
                sb.append("Requested ViewType is FooterView type. So, you have to use another type value.");
            }
            throw new IllegalArgumentException(TAG + sb.toString());
        }
        mViewBinderList.put(viewType, viewBinder);
        return true;
    }

    public boolean removeViewBinder(int viewType) {
        ViewBinder<T> element;
        final int size = mViewBinderList.size();
        for (int i = 0; i < size; i++) {
            element = mViewBinderList.valueAt(i);
            if (element.getItemViewType() == viewType) {
                mViewBinderList.removeAt(i);
                Log.d(TAG, LOG_PREFIX + "Removed the ViewBinder. type : " + viewType);
                return true;
            }
        }
        Log.d(TAG, LOG_PREFIX + "Failed to remove the ViewBinder. (There is no view binder) type : " + viewType);
        return false;
    }

    public boolean existViewBinder(ViewBinder<T> viewBinder) {
        RecyclerViewUtils.checkNotNull(viewBinder, "ViewBinder is null. You must pass a valid object.");

        final int viewType = viewBinder.getItemViewType();
        return mViewBinderList.get(viewType) != null;
    }

    /**
     * Return the view type of the item at <code>position</code> for the purposes
     * of view recycling.
     * <p/>
     * <p>The default implementation of this method returns 0, making the assumption of
     * a single view type for the adapter. Unlike ListView adapters, types need not
     * be contiguous. Consider using id resources to uniquely identify item view types.
     *
     * @param dataSet  data source / items
     * @param position position to query
     *
     * @return integer value identifying the type of the view needed to represent the item at
     * <code>position</code>.
     */
    public int getItemViewType(@NonNull T dataSet, int position) {
        RecyclerViewUtils.checkNotNull(dataSet, "DataSet is null. You must pass a valid dataSet.");

        ViewBinder<T> element;
        final int size = mViewBinderList.size();
        for (int i = 0; i < size; i++) {
            element = mViewBinderList.valueAt(i);
            if (element.isForViewType(dataSet, position)) {
                return element.getItemViewType();
            }
        }

        throw new IllegalArgumentException(
                "No ViewBinder added that matches position =" + position +
                        " in data source.\nViewBinder Size is " + mViewBinderList.size());
    }

    @NonNull
    public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        final ViewBinder<T> viewBinder = mViewBinderList.get(viewType);
        RecyclerViewUtils.checkNotNull(viewBinder, "No ViewBinder added for ViewType " + viewType);

        return viewBinder.onCreateViewHolder(parent);
    }

    public void onBindViewHolder(@NonNull T items, int position, @NonNull RecyclerView.ViewHolder viewHolder) {
        ViewBinder<T> viewBinder = mViewBinderList.get(viewHolder.getItemViewType());
        if (viewBinder == null) {
            throw new NullPointerException(
                    "No ViewBinder added for ViewType " + viewHolder.getItemViewType());
        }
        viewBinder.onBindViewHolder(items, position, viewHolder);
    }

    public void clear() {
        if (mViewBinderList.size() == 0) {
            return;
        }
        mViewBinderList.clear();
    }

    // ===========================================================
    // Listeners
    // ===========================================================

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================
}
