/*
 * Copyright 2015 Kim Seong-il
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kr.seongil.recyclerview.viewbinder;

import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;

import java.util.List;

import kr.seongil.recyclerview.model.common.RecyclerViewItem;

/**
 * I refer to follow project and customized it.
 * https://github.com/sockeqwe/AdapterDelegates/
 * <p/>
 * A simple {@link ViewBinder} implementation that already implements {@link #getItemViewType()}
 * Also, have a item click listener and long click listener.
 * Both of listeners will be moved into outer class from the {@link ViewBinder} in the next refactoring.
 *
 * @author Kim Seong-il
 * @since 1.0.0
 */
public abstract class AbstractViewBinder<T extends List> implements ViewBinder<T> {

    // ===========================================================
    // Constants
    // ===========================================================
    /**
     * Constant indicating the header view type. {@link AbstractHeaderViewBinder} will use this constant.
     * Also, {@link ViewBinderListManager} can have a one item having this view type.
     */
    public static final int RECYCLER_HEADER_VIEW_TYPE = 1000;

    /**
     * Constant indicating the footer view type. {@link AbstractFooterViewBinder} will use this constant.
     * Also, {@link ViewBinderListManager} can have a one item having this view type.
     */
    public static final int RECYCLER_FOOTER_VIEW_TYPE = 1001;

    // ===========================================================
    // Fields
    // ===========================================================
    /**
     * The item view type
     */
    protected int mViewType;

    /**
     * Click listener for the RecyclerView item.
     */
    protected RecyclerViewItemClickListener mItemViewClickListener;

    /**
     * Long click listener for the RecyclerView item.
     */
    protected RecyclerViewItemLongClickListener mItemViewLongClickListener;

    /**
     * Layout inflater.
     */
    protected LayoutInflater mLayoutInflater;

    // ===========================================================
    // Constructors
    // ===========================================================
    public AbstractViewBinder(int viewType, LayoutInflater inflater) {
        this(viewType, inflater, null, null);
    }

    public AbstractViewBinder(int viewType, LayoutInflater inflater, RecyclerViewItemClickListener itemClickListener) {
        this(viewType, inflater, itemClickListener, null);
    }

    public AbstractViewBinder(int viewType, LayoutInflater inflater, RecyclerViewItemLongClickListener itemLongClickListener) {
        this(viewType, inflater, null, itemLongClickListener);
    }

    public AbstractViewBinder(int viewType, LayoutInflater inflater, RecyclerViewItemClickListener itemClickListener,
                              RecyclerViewItemLongClickListener itemLongClickListener) {
        mViewType = viewType;
        mLayoutInflater = inflater;
        mItemViewClickListener = itemClickListener;
        mItemViewLongClickListener = itemLongClickListener;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================
    @Override
    public int getItemViewType() {
        return mViewType;
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Listeners
    // ===========================================================

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    /**
     * Interface definition for a callback invoked when user is clicked the {@link RecyclerViewItem}
     *
     * @author Kim Seong-il
     * @since 1.0.0
     */
    public interface RecyclerViewItemClickListener<T, V extends RecyclerView.ViewHolder> {

        // ===========================================================
        // Constants
        // ===========================================================

        // ===========================================================
        // Methods
        // ===========================================================
        void onClickedRecyclerViewItem(V vh, T info, int position);
    }

    /**
     * Interface definition for a callback invoked when user is long-clicked the {@link RecyclerViewItem}
     *
     * @author Kim Seong-il
     * @since 1.0.0
     */
    public interface RecyclerViewItemLongClickListener<T, V extends RecyclerView.ViewHolder> {

        // ===========================================================
        // Constants
        // ===========================================================

        // ===========================================================
        // Methods
        // ===========================================================
        void onLongClickedRecyclerViewItem(V vh, T info, int position);
    }
}
