/*
 * Copyright 2015 Kim Seong-il
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kr.seongil.recyclerview.viewadapter;

import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.ViewGroup;

import java.util.List;

import kr.seongil.recyclerview.model.common.RecyclerViewItem;
import kr.seongil.recyclerview.model.footerview.RecyclerViewFooterItem;
import kr.seongil.recyclerview.model.headerview.RecyclerViewHeaderItem;
import kr.seongil.recyclerview.utils.RecyclerViewUtils;
import kr.seongil.recyclerview.viewbinder.AbstractViewBinder;
import kr.seongil.recyclerview.viewbinder.ViewBinder;
import kr.seongil.recyclerview.viewbinder.ViewBinderListManager;

/**
 * I refer to follow project and customized it.
 * https://github.com/sockeqwe/AdapterDelegates
 * An implementation of an Adapter that already uses a {@link ViewBinderListManager} and calls the corresponding {@link ViewBinderListManager}
 * methods from Adapter's method like {@link #onCreateViewHolder(ViewGroup, int)}, {@link #onBindViewHolder(RecyclerView.ViewHolder, int)},
 * and {@link #getItemViewType(int)}. So everything is already setup for you. You just have to add the {@link ViewBinderListManager}s.
 * i.e) in the constructor of a subclass that inheritance from this class:
 * <pre>
 *     class SimpleConcreteAdapter extends AbstractRecyclerViewAdapter<ConcreteRecyclerViewItem>{
 *
 *         public SimpleConcreteAdapter(){
 *             this.addViewBinder(new ItemViewBinder1());
 *             this.addViewBinder(new ItemViewBinder2());
 *             this.addViewBinder(new ItemViewBinder3());
 *         }
 *     }
 * </pre>
 * Some member variables can be accessed by the multiple threads.
 * i.e) Client view is using AutoLoading with Header and Footer sides.
 * To prevent a synchronization issues, all of methods is applied the "synchronized (this)"
 *
 * @param <T> The type of {@link RecyclerViewItem}
 *
 * @author Kim Seong-il
 * @since 1.0.0
 */

public abstract class AbstractRecyclerViewAdapter<T extends RecyclerViewItem>
        extends RecyclerView.Adapter
        implements RecyclerListViewListener {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================
    private ViewBinderListManager<List<T>> mViewBinderManager = new ViewBinderListManager<>();
    protected LayoutInflater mLayoutInflater;
    protected List<T> mDataSet;

    // ===========================================================
    // Constructors
    // ===========================================================
    public AbstractRecyclerViewAdapter(LayoutInflater layoutInflater) {
        mLayoutInflater = layoutInflater;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * Get the items / data source of this adapter
     *
     * @return The items / data source
     */
    public synchronized List<T> getDataSet() {
        return mDataSet;
    }

    /**
     * Set the items / data source of this adapter
     *
     * @param dataSet the items / data source
     */
    public synchronized void setDataSet(List<T> dataSet) {
        mDataSet = dataSet;
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================
    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        synchronized (this) {
            return mViewBinderManager.onCreateViewHolder(parent, viewType);
        }
    }

    @Override
    public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
        synchronized (this) {
            mViewBinderManager.onBindViewHolder(mDataSet, position, holder);
        }
    }

    @Override
    public int getItemViewType(int position) {
        synchronized (this) {
            return mViewBinderManager.getItemViewType(mDataSet, position);
        }
    }

    @Override
    public int getItemCount() {
        synchronized (this) {
            return mDataSet == null ? 0 : mDataSet.size();
        }
    }

    // ===========================================================
    // Methods
    // ===========================================================

    /**
     * Clear all the added view binders from the list.
     */
    protected synchronized void clearViewBinders() {
        mViewBinderManager.clear();
    }

    /**
     * Add a view binder to the list.
     *
     * @param cvb Target view binder
     *
     * @return
     */
    @SuppressWarnings("unchecked")
    protected synchronized boolean addViewBinder(AbstractViewBinder cvb) {
        return mViewBinderManager.addViewBinder(cvb);
    }

    /**
     * Remove a view binder from the list as per the given view type.
     *
     * @param viewType ViewBinder type to remove item from the list
     *
     * @return
     */
    protected synchronized boolean removeViewBinder(int viewType) {
        return mViewBinderManager.removeViewBinder(viewType);
    }

    /**
     * Checks whether the given viewBinder exists or not.
     *
     * @param viewBinder Target ViewBinder to check the existence.
     *
     * @return
     */
    @SuppressWarnings("unchecked")
    public synchronized boolean existViewBinder(ViewBinder viewBinder) {
        return mViewBinderManager.existViewBinder(viewBinder);
    }

    public synchronized void addFirstCollection(List<T> collection) {
        RecyclerViewUtils.checkNotNull(collection, "Collection to add on the DataSet is null.");

        final int targetSize = collection.size();
        final int insertPos = registeredHeaderView() ? 1 : 0;
        mDataSet.addAll(insertPos, collection);
        notifyItemRangeInserted(-targetSize, targetSize);
    }

    public synchronized void addLastCollection(List<T> collection) {
        RecyclerViewUtils.checkNotNull(collection, "Collection to add on the DataSet is null.");

        final int targetSize = collection.size();
        int insertPos = getItemCount();

        if (registeredFooterView()) {
            if (getItemCount() == 1) {
                insertPos = 0;
            } else if (getItemCount() > 1) {
                insertPos = getItemCount() - 1;
            }
        }
        mDataSet.addAll(insertPos, collection);
        notifyItemRangeInserted(insertPos + 1, targetSize);
    }

    public synchronized void addLast(T object) {
        int insertionPos;
        if (!registeredFooterView()) {
            insertionPos = getItemCount();
            mDataSet.add(insertionPos, object);
            notifyItemInserted(insertionPos);
            return;
        }

        insertionPos = getItemCount() == 0 ? 0 : getItemCount() - 1;
        mDataSet.add(insertionPos, object);
        notifyItemInserted(insertionPos);
    }

    public synchronized void addFirst(T object) {
        if (!registeredHeaderView()) {
            mDataSet.add(0, object);
            notifyItemInserted(0);
            return;
        }

        final int insertionPos = getItemCount() == 0 ? 0 : 1;
        mDataSet.add(insertionPos, object);
        notifyItemInserted(insertionPos);
    }

    public synchronized void addPosition(T object, int position) {
        mDataSet.add(position, object);
        notifyItemInserted(position);
    }

    public synchronized void removeLastItem() {
        if (getItemCount() == 0) {
            return;
        }

        final int removeItemPos = getItemCount() - 1;
        mDataSet.remove(removeItemPos);
        notifyItemRemoved(removeItemPos);
    }

    public synchronized void removeFirstItem() {
        if (getItemCount() == 0) {
            return;
        }
        mDataSet.remove(0);
        notifyItemRemoved(0);
    }

    public synchronized void updatePositionWithNotify(T object, int position) {
        updatePosition(object, position);
        notifyItemChanged(position);
    }

    public synchronized void updatePosition(T object, int position) {
        mDataSet.set(position, object);
    }

    public synchronized void clearDataSet() {
        final int size = getItemCount();
        mDataSet.clear();
        notifyItemRangeRemoved(0, size);
    }

    protected void removeFooterItem() {
        if (!registeredFooterView()) {
            throw new IllegalArgumentException("Adapter is not using the footer view");
        }
        if (getItemCount() == 0) {
            throw new IndexOutOfBoundsException("There is no items in the DataSet");
        }
        if (!(mDataSet.get(getItemCount() - 1) instanceof RecyclerViewFooterItem)) {
            throw new RuntimeException("The last item of the DataSet MUST be an instance of RecyclerViewFooterItem");
        }

        removeLastItem();
    }

    protected void removeHeaderItem() {
        if (!registeredHeaderView()) {
            throw new IllegalArgumentException("Adapter is not using the header view");
        }
        if (getItemCount() == 0) {
            throw new IndexOutOfBoundsException("There is no items in the DataSet");
        }
        if (!(mDataSet.get(0) instanceof RecyclerViewHeaderItem)) {
            throw new RuntimeException("The first item of the DataSet MUST be an instance of RecyclerViewHeaderItem");
        }

        removeFirstItem();
    }

    public synchronized void setData(int position, T object) {
        if (getItemCount() <= position) {
            throw new IllegalArgumentException("Position is invalid");
        }
        mDataSet.set(position, object);
        notifyItemChanged(position);
    }

    // ===========================================================
    // Listeners
    // ===========================================================

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================
}
