/*
 * Copyright 2015 Kim Seong-il
 *
 * Licensed under the Apache License, Version 2.0 (the "License")
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kr.seongil.recyclerview.model.common;

import kr.seongil.recyclerview.model.footerview.RecyclerViewFooterItem;
import kr.seongil.recyclerview.model.headerview.RecyclerViewHeaderItem;

/**
 * This is kind of entity class having information related with Header and Footer item.
 * This entity MUST be used by {@link RecyclerViewHeaderItem} or {@link RecyclerViewFooterItem} only.
 * <p/>
 * And, this class having a status code that represents the concrete class's status.
 * i.e) Show the loading view. Show the label view.
 *
 * @author Kim Seong-il
 * @since 1.0.0
 */
public class RecyclerViewHeaderFooterItem implements RecyclerViewItem {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================
    /**
     * Represents the state of the concrete class.
     * If {#applyOneCycle} is true, follows the {#statusCodeForOneCycle} value.
     */
    private Status statusCode = Status.IDLE;

    /**
     * Represents the state of the concrete class.
     * This status code is valid when {#applyOneCycle} got "true".
     */
    private Status statusCodeForOneCycle = Status.IDLE;
    private boolean applyOneCycle = false;

    // ===========================================================
    // Constructors
    // ===========================================================
    public RecyclerViewHeaderFooterItem() {
    }

    public RecyclerViewHeaderFooterItem(Status statusCode) {
        this.statusCode = statusCode;
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================
    public Status getStatusCode() {
        return applyOneCycle ? statusCodeForOneCycle : statusCode;
    }

    public boolean isApplyOneCycle() {
        return applyOneCycle;
    }

    public void setApplyOneCycle(boolean applyOneCycle) {
        this.applyOneCycle = applyOneCycle;
    }

    public void setStatusCode(Status statusCode) {
        this.statusCode = statusCode;
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    // ===========================================================
    // Methods
    // ===========================================================
    public void setStatusCode(Status statusCode, boolean applyOneCycle) {
        this.applyOneCycle = applyOneCycle;
        if (applyOneCycle) {
            this.statusCodeForOneCycle = statusCode;
        } else {
            this.statusCode = statusCode;
        }
    }

    public boolean autoLoadData() {
        return statusCode == Status.VISIBLE_LOADING_VIEW;
    }

    // ===========================================================
    // Listeners
    // ===========================================================

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    /**
     * Enumerates the status codes.
     */
    public enum Status {

        // ===========================================================
        // Fields
        // ===========================================================
        IDLE,
        VISIBLE_LOADING_VIEW,
        VISIBLE_LABEL_VIEW,
        HIDE_VIEW;

        // ===========================================================
        // Constructors
        // ===========================================================
        Status() {
        }
    }
}