/*
 * Copyright 2015 Kim Seong-il
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kr.seongil.recyclerview.listener;

import android.support.annotation.NonNull;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.util.Log;

import kr.seongil.recyclerview.utils.RecyclerViewUtils;
import kr.seongil.recyclerview.viewadapter.RecyclerListViewListener;

/**
 * @author Kim Seong-il
 * @since 1.0.0
 */
public class RecyclerViewScrollListener extends RecyclerView.OnScrollListener {

    // ===========================================================
    // Constants
    // ===========================================================
    private static final String TAG = RecyclerViewUtils.TAG;
    private static final String LOG_PREFIX = RecyclerViewUtils.getPrefix("RecyclerViewScrollListener");

    /**
     * Represents the count of loaded items in the {@link RecyclerView}
     */
    private static final int CNT_OF_MINIMUM_LOADED_ITEMS = 20;

    // ===========================================================
    // Fields
    // ===========================================================
    /**
     * Used by detecting the timing to load items more.
     */
    private LinearLayoutManager mLinearLayoutManager;
    private RecyclerListViewListener mRecyclerListViewListener;
    private final int mLoadingItemsThreshold;

    private int mCntOfPrevTotalItem;
    private int mPosOfFirstVisibleItem, mCntOfVisibleItem, mCntOfTotalItem;

    // ===========================================================
    // Constructors
    // ===========================================================
    public RecyclerViewScrollListener(
            @NonNull final LinearLayoutManager llm,
            @NonNull final RecyclerListViewListener recyclerListViewListener,
            final int loadingThreshold) {

        mLinearLayoutManager = llm;
        mRecyclerListViewListener = recyclerListViewListener;

        if (loadingThreshold <= 0) {
            throw new IllegalArgumentException("Threshold is invalid. You must pass it greater than 0.");
        }
        mLoadingItemsThreshold = loadingThreshold;

        initScrollListener();
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================
    @Override
    public void onScrolled(RecyclerView recyclerView, int dx, int dy) {
        super.onScrolled(recyclerView, dx, dy);
        Direction scrollDirection = decideScrollDirection(dx, dy);

        mCntOfVisibleItem = recyclerView.getChildCount();
        mCntOfTotalItem = mLinearLayoutManager.getItemCount();
        mPosOfFirstVisibleItem = mLinearLayoutManager.findFirstVisibleItemPosition();

        if (scrollDirection == Direction.UP || scrollDirection == Direction.LEFT) {
            handleScrollUp();
        } else if (scrollDirection == Direction.DOWN || scrollDirection == Direction.RIGHT) {
            handleScrollDown();
        } else if (scrollDirection == Direction.STOP) {
            loadItemsMoreForcibly();
        } else {
            throw new IllegalArgumentException("Undefined scrolling direction.");
        }
    }

    // ===========================================================
    // Methods
    // ===========================================================
    public void initScrollListener() {
        mCntOfPrevTotalItem = 0;
        mPosOfFirstVisibleItem = 0;
        mCntOfVisibleItem = 0;
        mCntOfTotalItem = 0;
    }

    public void loadItemsMoreForcibly() {
        if (mCntOfTotalItem >= CNT_OF_MINIMUM_LOADED_ITEMS) {
            // We don't need to load more data forcibly.
            return;
        }

        if (mRecyclerListViewListener.registeredHeaderView()) {
            onLoadPrevData();
        }
        if (mRecyclerListViewListener.registeredFooterView()) {
            onLoadNextData();
        }

        Log.d(TAG, LOG_PREFIX + "The count of items in LayoutManager is "
                + mCntOfTotalItem + ". So we are need to load more data forcibly.");
    }

    /**
     * Is the client view loading "next" items currently?
     * If you override {@link # onLoadPrevData}, you MUST override this method also.
     *
     * @return true, the client view is loading the "next" items currently. otherwise, false
     */
    protected boolean isLoadingNextItems() {
        return false;
    }

    /**
     * Is the client view loading "previous" items currently?
     * If you override {@link # onLoadPrevData}, you MUST override this method also.
     *
     * @return true, the client view is loading the "previous" items currently. otherwise, false
     */
    protected boolean isLoadingPrevItems() {
        return false;
    }

    /**
     * If you want to load the next data more, override this method.
     */
    public void onLoadNextData() {
        // Do nothing
    }

    /**
     * If you want to load previous data more, override this method.
     */
    public void onLoadPrevData() {
        // Do nothing
    }

    /**
     * This method is triggered when the {@link RecyclerView} is scrolling to the top position.
     */
    private void handleScrollDown() {
        if (!isLoadingPrevItems()
                && mRecyclerListViewListener.loadHeaderItemsMore()
                && (mPosOfFirstVisibleItem <= mLoadingItemsThreshold)) {
            onLoadPrevData();
        }
    }

    /**
     * This method is triggered when the {@link RecyclerView} is scrolling to the bottom position.
     */
    private void handleScrollUp() {
        if (isLoadingNextItems() && mCntOfTotalItem > mCntOfPrevTotalItem) {
            mCntOfPrevTotalItem = mCntOfTotalItem;
        }

        if (!isLoadingNextItems()
                && mRecyclerListViewListener.loadFooterItemsMore()
                && (mCntOfTotalItem - mCntOfVisibleItem) <= (mPosOfFirstVisibleItem + mLoadingItemsThreshold)) {
            onLoadNextData();
        }
    }

    /**
     * Decide the scrolling direction.
     */
    private Direction decideScrollDirection(final int dx, final int dy) {
        if (dx > 0) {
            return Direction.RIGHT;
        } else if (dx < 0) {
            return Direction.LEFT;
        }
        if (dy > 0) {
            return Direction.UP;
        } else if (dy < 0) {
            return Direction.DOWN;
        }
        return Direction.STOP;
    }

    // ===========================================================
    // Listeners
    // ===========================================================

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================
    private enum Direction {
        STOP,
        RIGHT,
        LEFT,
        UP,
        DOWN;

        Direction() {
        }
    }
}
