/*
 * Copyright (c) 2020-2024 IBA Group.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v2.0 which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-v20.html
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   IBA Group
 *   Zowe Community
 */

package org.zowe.kotlinsdk.zowe.client.sdk.zostso

import org.zowe.kotlinsdk.*
import org.zowe.kotlinsdk.zowe.client.sdk.core.ZOSConnection
import org.zowe.kotlinsdk.zowe.client.sdk.zostso.input.StartTsoParams
import okhttp3.Credentials
import okhttp3.OkHttpClient
import retrofit2.Response

/**
 * Start TSO address space and receive servlet key
 */
class StartTso(
  var connection: ZOSConnection,
  var httpClient: OkHttpClient = UnsafeOkHttpClient.unsafeOkHttpClient
) {

  init {
    connection.checkConnection()
  }

  var response: Response<*>? = null

  /**
   * Start TSO address space with provided parameters.
   *
   * @param accountNumber this key of StartTsoParams required, because it cannot be default.
   * @param params        optional [StartTsoParams] object with required parameters
   * @return [StartStopResponses] command response on resolve
   * @throws Exception error executing command
   */
  @Throws(Exception::class)
  fun start(accountNumber: String, params: StartTsoParams, failOnPrompt: Boolean): StartStopResponses {
    if (accountNumber.isEmpty()) {
      throw Exception("accountNumber not specified")
    }

    val customParams = StartTsoParams(
      account = accountNumber,
      characterSet = params.characterSet ?: TsoConstants.DEFAULT_CHSET,
      codePage = params.codePage ?: TsoConstants.DEFAULT_CPAGE,
      rows = params.rows ?: TsoConstants.DEFAULT_ROWS,
      columns = params.columns ?: TsoConstants.DEFAULT_COLS,
      logonProcedure = params.logonProcedure ?: TsoConstants.DEFAULT_PROC,
      regionSize = params.regionSize ?: TsoConstants.DEFAULT_RSIZE,
    )

    val tsoResponse = startCommon(customParams)

    var collectedResponses: CollectedResponses? = null
    if (tsoResponse.servletKey != null) {
      val sendTso = SendTso(connection, httpClient)
      collectedResponses = sendTso.getAllResponses(tsoResponse, failOnPrompt)
    }

    return StartStopResponses(tsoResponse, collectedResponses)
  }

  /**
   * Start TSO address space with provided parameters
   *
   * @param commandParams [StartTsoParams] object with required parameters
   * @return [TsoResponse] z/OSMF response object
   * @throws Exception error executing command
   */
  @Throws(Exception::class)
  fun startCommon(commandParams: StartTsoParams): TsoResponse {
    val url = "${connection.protocol}://${connection.host}:${connection.zosmfPort}"
    val tsoApi = buildApi<TsoApi>(url, httpClient)
    val call = tsoApi.startTso(
      authorizationToken = Credentials.basic(connection.user, connection.password),
      acct = commandParams.account ?: throw Exception("account num not specified"),
      proc = commandParams.logonProcedure ?: TsoConstants.DEFAULT_PROC,
      chset = commandParams.characterSet ?: TsoConstants.DEFAULT_CHSET,
      cpage = commandParams.codePage ?: TsoConstants.DEFAULT_CPAGE,
      rows = commandParams.rows?.toInt() ?: TsoConstants.DEFAULT_ROWS.toInt(),
      cols = commandParams.columns?.toInt() ?: TsoConstants.DEFAULT_COLS.toInt(),
      rsize = commandParams.regionSize?.toInt() ?: TsoConstants.DEFAULT_RSIZE.toInt()
    )
    response = call.execute()
    validateResponse(response, "No results from executing tso command while setting up TSO address space")
    return response?.body() as TsoResponse? ?: throw Exception("No body returned")
  }
}
